
/* GraphCanvas.java
 * Copyright (c) 1997 Bill Bereza. All Rights Reserved.
 *
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    Contact Bill Bereza:
    email: bereza@pobox.com
    url:   www.pobox.com/~bereza
    
    address:
           9526 Judson Rd.
           Ravenna, MI 49451
*/

import Values;
import InfoFrame;
import Filters;
import QSortAlgorithm;
import OptionsFrame;
import OptionTaking;

import java.awt.*;
import java.applet.*;
import java.util.*;
import java.net.*;
import java.io.*;

class Legend {
    public Rectangle rect;
    public String description;
    public boolean shown;
    
    Legend() {
        shown=false;
    }
}

public class GraphCanvas extends Canvas implements OptionTaking {
    protected Values myParser;
    protected String type;
    protected String options;
    protected boolean doGrid;
    protected Color gridColor;
    protected Color graphColor;
    protected Color backColor;
    protected Color pieColors[];
    protected int maxPies;
    protected double xGridEvery;
    protected double yGridEvery;
    protected Color fontColor;
    protected Font legendFont;
    protected boolean changed; // true if something has changed which should cause the entire graph to be redrawn
    protected Object optioner;
    protected Vector legends;
    
    private LogFrame logFrame;
    private boolean optionerShown;
    private boolean loggerShown;
    private TextField statusField;
    
    final public String pieType="pie";
    final public String lineType="line";
    final public String barType="bar";

    public void setStatusReciever(TextField to) { statusField=to; }

    public void setParser(Values to) {
        myParser=to;
        options=" Accesses Weekly "; //myParser.accesses+myParser.weekly;
    }

    GraphCanvas() {
        changed=true;
        optionerShown=false;
        myParser=null;
        type=lineType;
        //options=WWWParser.accesses+WWWParser.weekly;
        doGrid=true;
        maxPies=10;
        pieColors=new Color[maxPies];
        pieColors[0]=Color.blue;
        pieColors[1]=Color.cyan;
        pieColors[2]=Color.lightGray;
        pieColors[3]=Color.green;
        pieColors[4]=Color.magenta;
        pieColors[5]=Color.orange;
        pieColors[6]=Color.pink;
        pieColors[7]=Color.red;
        pieColors[8]=Color.yellow;
        pieColors[9]=Color.darkGray;
        graphColor=Color.red;
        backColor=new Color(255,255,240);
        gridColor=new Color(0,0,240);
        legendFont=new Font("Helvetica",Font.ITALIC+Font.BOLD,10);
        fontColor=Color.black;
        optionerShown=false;
        
        xGridEvery=0.0;
        yGridEvery=1.0;
        
        logFrame=new LogFrame();
        loggerShown=false;
        optioner=new OptionsFrame(this);
        legends=new Vector();
        //repaint();
    }
    
    public void setOptioner(Object to) { optioner=to; }
    public Object getOptioner() { return optioner; }
    public void doOptioner() {
        ((OptionsFrame)optioner).show();
        if(!optionerShown)
            ((OptionsFrame)optioner).InitialPositionSet();
        optionerShown=true;
    }
    
    public void setChanged(boolean to) { changed=to; if(to) repaint(); }
    public boolean getChanged() { return changed; }
    
    public void setXGridEveryOther(double to) { xGridEvery=to; setChanged(true); }
    public void setYGridEveryOther(double to) { yGridEvery=to; setChanged(true); }
    public double getXGridEveryOther() { return xGridEvery; }
    public double getYGridEveryOther() { return yGridEvery; }
    
    public void setPieColors(Color to[]) { pieColors=to; setChanged(true);}
    public Color[] getPieColors() { return pieColors; }
    
    public void setBackGroundColor(Color to) { backColor=to; setChanged(true);}
    public Color getBackGroundColor() { return backColor; }
    
    public void setGraphColor(Color to) { graphColor=to; setChanged(true);}
    public Color getGraphColor() { return graphColor; }
    
    public void setGridColor(Color to) { gridColor=to; setChanged(true); }
    public Color getGridColor() { return gridColor; }
    
    public void setMaximumPies(int to) { maxPies=to; setChanged(true); }
    public int getMaximumPies() { return maxPies; }
    
    public void setDoGrid(boolean to) { doGrid=to; setChanged(true); }
    public boolean getDoGrid() { return doGrid; }
    
    public void setType(String to) { type=to; setChanged(true); }
    public String getType() { return type; }
    
    public void setOptions(String to) { options=to; setChanged(true); }
    public String getOptions() { return options; }

    public void setFontColor(Color to) { fontColor=to; setChanged(true);}
    public Color getFontColor() { return fontColor; }
    
    public void setLegendFont(Font to) { legendFont=to; setChanged(true);}
    public Font getLegendFont() { return legendFont; }    
    
    protected void paintGrid(Graphics g, int x, int y, int width, int height, double wScale, double hScale) {
        double loop;
        double xEv, yEv;
        
        xEv=getXGridEveryOther();
        yEv=getYGridEveryOther();
        
        g.setColor(getGridColor());
        
        g.drawRect(x,y,width,height);
        
        for(loop=((double)(width));loop>0;loop=loop-wScale-(xEv*wScale)) {
            //System.out.println("drawLine("+(int)loop+","+y+","+(int)loop+","+height+")");
            g.drawLine((int)Math.round(loop), y, (int)Math.round(loop), height);
        }
        
        for(loop=((double)(height));loop>0;loop=loop-hScale-(yEv*hScale))
            g.drawLine(x, (int)Math.round(loop), width, (int)Math.round(loop));
    
    }

    //public void update(Graphics g) { System.out.println("update canvas"); paint(g); }

    public void paint(Graphics g) { 
        //System.out.println("painting canvas: "+(new Date()).toString());
        
        if(myParser==null)
            return;
        legends.removeAllElements();
        // empty logFrame.logArea when I changed this to JDK1.1
        if(getType().equals(pieType)) {
            paintPie(g);
        }
        else if(getType().equals(lineType)){
            paintLines(g); 
        }
        else {
            paintBars(g);
        }
    }

    protected void paintPie(Graphics g) {
        Rectangle r=bounds();
        Vector points;
        Value vals[];
        QSortAlgorithm sorter;
        CompareValues comparer;
        int loop;
        double totalValue;
        int x, y, axLen, lastArc, arc, pies;
        int legendY, size;
        int shownValue;
        int fontSize=legendFont.getSize();
        FontMetrics tMet;
        Legend leg;
        
        if(myParser.getRecount()) { //parser had to recount values
            changed=true;
        }
        
        if(getChanged()) {
            g=getGraphics();
            //g.setClip(0, 0, r.width, r.height);
            setChanged(false);
        }

        g.setColor(getBackGroundColor());
        g.fillRect(0,0,r.width,r.height);
        
        if(r.width>r.height) {
            y=0;
            x=r.width-r.height;
            axLen=r.height-1;
        } 
        else {
            x=0;
            y=r.height-r.width;
            axLen=r.width-1;
        }
        
        sorter=new QSortAlgorithm();
        comparer=new CompareValues();
        
        points=myParser.getValues(options);
        size=points.size();
        
        vals=new Value[size];
        points.copyInto(vals);
        try {
            sorter.sort(vals,comparer);
        } catch(Exception e) {
            System.out.println(e.getMessage());
        }
        
        totalValue=0.0;
        pies=(size<maxPies) ? size : maxPies;
        //System.out.println("pies: "+pies+" "+myParser.getMaxValue());
        for(loop=0;loop<size;loop++) {
            totalValue+=(double)vals[loop].value;
        }
        
        tMet=getFontMetrics(legendFont);
        fontSize=tMet.getHeight();
        g.setFont(legendFont);
        lastArc=0; legendY=0; shownValue=0;
        for(loop=0;loop<pies;loop++) {
            arc=(int)Math.round((((double)(vals[size-loop-1].value))/totalValue)*360.0);
            g.setColor(pieColors[loop%10]);
            
            g.fillArc(x, y, axLen, axLen, lastArc, arc);
            lastArc+=arc;
        }
        g.setColor(Color.black);
        g.drawArc(x,y,axLen,axLen,0,360);
        for(loop=0;loop<pies;loop++) {
            leg=new Legend();
            leg.rect=new Rectangle(0, legendY, tMet.stringWidth(((String)vals[size-loop-1].description)+" - "+Integer.toString(vals[size-loop-1].value)), fontSize);
            leg.description=((String)vals[size-loop-1].description);
            legends.addElement(leg);
            g.setColor(pieColors[loop%10]);
            g.fillRect(0,legendY,fontSize,fontSize);
            g.setColor(fontColor);
            g.drawRect(0,legendY,fontSize,fontSize);
            g.drawString(((String)vals[size-loop-1].description)+" - "+Integer.toString(vals[size-loop-1].value),fontSize+4,legendY+fontSize);
            legendY+=(fontSize+2);
            shownValue+=vals[size-loop-1].value;
        }
        if(shownValue<(int)totalValue) {
            g.setColor(fontColor);
            g.drawRect(0,legendY,fontSize,fontSize);
            g.drawString("Other - "+Integer.toString((int)totalValue-shownValue),fontSize+4,legendY+fontSize);
        }
        g.setColor(fontColor);
        g.drawString("Total Accesses: "+Integer.toString((int)totalValue),0,r.height-1-tMet.getDescent());
    }

    public boolean mouseDown(Event evt, int x, int y) {
        boolean showit;
        Legend d0;
        Vector trans;

        showit=false;
        d0=null;
        for (Enumeration e = legends.elements() ; !showit && e.hasMoreElements() ;) {
            Legend el;
            el=(Legend)(e.nextElement());
            if(x>=el.rect.x && x<=el.rect.x+el.rect.width && y>=el.rect.y && y<=el.rect.y+el.rect.height) { // isInside
                showit=true;
                d0=el;
            }
        }
        if(!showit) {
            return false;
        }
        if(!d0.shown) {
           trans=myParser.getTransfers(options,d0.description);
           logFrame.show();
           if(!loggerShown)
            logFrame.InitialPositionSet();
           loggerShown=true;           
           logFrame.logArea.appendText("=== Transfers of "+options+" for "+d0.description+" -------------------\n");
           for (Enumeration e = trans.elements() ; e.hasMoreElements() ;) {            
               WWWTransfer el;
               el=(WWWTransfer)(e.nextElement());
               logFrame.logArea.appendText(el.toString()+"\n");
           }
        }
        logFrame.show();
        return true;
    }
    
    public boolean mouseMove(Event evt, int x, int y) {
        boolean showit;
        Legend d0;
        Vector trans;

        showit=false;
        d0=null;
        for (Enumeration e = legends.elements() ; !showit && e.hasMoreElements() ;) {
            Legend el;
            el=(Legend)(e.nextElement());
            if(x>=el.rect.x && x<=el.rect.x+el.rect.width && y>=el.rect.y && y<=el.rect.y+el.rect.height) { // isInside
                showit=true;
                d0=el;
            }
        }
        if(!showit) {
            return false;
        }
        statusField.setText(d0.description);
        return true;
    }

    protected void paintLines(Graphics g) {
        Rectangle r=bounds();
        double wscale, hscale, x1,y1,x2,y2;
        Value el;
        Vector points;
        double loop;
        FontMetrics tMet;
        int t;

        if(myParser.getRecount()) { //parser had to recount values
            changed=true;
        }
        
        if(getChanged()) {
            g=getGraphics();
            //g.setClip(0, 0, r.width, r.height);
            setChanged(false);
        }

        g.setColor(getBackGroundColor());
        g.fillRect(0,0,r.width,r.height);
        
        points=myParser.getValues(options);
        
        wscale=(double)(r.width)/(double)(points.size());
        hscale=(double)(r.height)/(double)(myParser.getMaxValue());
        
        if(doGrid) paintGrid(g, 0,0,r.width-1, r.height-1, wscale, hscale);
        
        //System.out.println("options: "+options+"\ntype: "+type+"\nwscale: "+wscale+" hscale: "+hscale);
        
        g.setColor(getGraphColor());
        x1=-1.0;
        y1=-1.0;
        //System.out.println("looping elements: "+points.size());
        loop=0.0;
        for (Enumeration e = points.elements() ; e.hasMoreElements() ;loop++) {
            el=(Value)(e.nextElement());
            x2=(loop*wscale); //x1+wscale;
            y2=(double)(r.height-1)-(double)(el.value)*hscale;
            if(x1<0.0) {
                x1=x2; y1=y2;
            }
            //System.out.println("plotting points ("+x1+","+y1+","+x2+","+y2+")");
            g.drawLine((int)x1,(int)y1,(int)x2,(int)y2);
            x1=x2; y1=y2;
        }
        //numValsLabel.setText(Integer.toString(myParser.getValues(graphPanel.myCanvas.getOptions()).size()));
        //maxValLabel.setText(Integer.toString(myParser.getMaxValue()));
        
        tMet=getFontMetrics(legendFont);
        g.setFont(legendFont);
        g.setColor(fontColor);
        t=(tMet.getMaxAdvance()<0) ? 1 : tMet.getMaxAdvance();
        g.drawString(Integer.toString(myParser.getMaxValue()),t,tMet.getMaxAscent()+1);
        g.drawString(Integer.toString(points.size()),r.width-1-tMet.stringWidth(Integer.toString(points.size()))-t,r.height-1-tMet.getMaxDescent());
    }
    protected void paintBars(Graphics g) {
        Rectangle r=bounds();
        double wscale, hscale, x1,y1,x2,y2;
        Value el;
        Vector points;
        double loop;
        FontMetrics tMet;
        int t;
        Value vals[];
        QSortAlgorithm sorter;
        CompareValues comparer;

        double totalValue;
        int x, y, axLen, lastArc, arc, pies;
        int legendY, size;
        int shownValue;
        int fontSize=legendFont.getSize();
        Legend leg;


        if(myParser.getRecount()) { //parser had to recount values
            changed=true;
        }
        
        if(getChanged()) {
            g=getGraphics();
            //g.setClip(0, 0, r.width, r.height);
            setChanged(false);
        }

        g.setColor(getBackGroundColor());
        g.fillRect(0,0,r.width,r.height);
        
        points=myParser.getValues(options);
              
        size=points.size();
        pies=(size<maxPies) ? size : maxPies;

        wscale=(double)(r.width)/(double)(points.size());
        hscale=(double)(r.height)/(double)(myParser.getMaxValue());
        
        if(doGrid) paintGrid(g, 0,0,r.width-1, r.height-1, wscale, hscale);       
      
        g.setColor(getGraphColor());
        x1=0.0;
        y1=(double)(r.height-1);
        loop=0.0;
        for (Enumeration e = points.elements() ; e.hasMoreElements() ;loop++) {
            el=(Value)(e.nextElement());
            x2=(loop*wscale); //x1+wscale;
            y1=(double)(r.height-1)-(double)(el.value)*hscale;
            y2=(double)(el.value)*hscale;

            g.setColor(pieColors[((int)loop)%10]);
            g.fillRect((int)x2,(int)y1,(int)wscale, (int)y2);
            g.setColor(Color.black);
            g.drawRect((int)x2,(int)y1,(int)wscale, (int)y2);
            
            x1=x2; // y1=y2;

            leg=new Legend();
            leg.rect=new Rectangle((int)x2,(int)y1,(int)wscale, (int)y2);
            leg.description=((String)(el.description));
            legends.addElement(leg);
        }
        
        tMet=getFontMetrics(legendFont);
        g.setFont(legendFont);
        g.setColor(fontColor);
        t=(tMet.getMaxAdvance()<0) ? 1 : tMet.getMaxAdvance();
        g.drawString(Integer.toString(myParser.getMaxValue()),t,tMet.getMaxAscent()+1);
        g.drawString(Integer.toString(points.size()),r.width-1-tMet.stringWidth(Integer.toString(points.size()))-t,r.height-1-tMet.getMaxDescent());
    }
}

